//
// IO-Warrior Winamp VU-meter visualization plugin
// Note: this plugin is for Winamp 2.xx
// It won't work with Winamp 3!
//
// Based on vis MiniSDK by Nullsoft, Inc.
//

#include <windows.h>
#include <math.h>
#include <stdlib.h>
#include "vis.h"
#include "iowkit.h"

char szAppName[] = "IOWarrior LEDs Visualization plugin";

// configuration declarations
// default screen X position and Y position, repsectively
int config_x = 50, config_y = 50;
// reads the configuration
void config_read(struct winampVisModule *mod);
// writes the configuration
void config_write(struct winampVisModule *mod);
// makes the .ini file filename
void config_getinifn(struct winampVisModule *mod, char *ini_file);

// returns a winampVisModule when requested. Used in hdr, below
winampVisModule *getModule(int which);

// Module functions
// configuration dialog
void Config(struct winampVisModule *mod);
// Module initialization
int Init(struct winampVisModule *mod);
// Render() (ie process and display data)
int Render(struct winampVisModule *mod);
// Module deinitialization, called when this vis plugin is unloaded
void Quit(struct winampVisModule *mod);

// Window procedure
LRESULT CALLBACK WndProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam);
// Main window handle
HWND hMainWnd;

// Double buffering data
HDC memDC;		// memory device context
// Double buffer for blitting
HBITMAP	memBM, oldBM;
// Is IO Warrior opened?
BOOLEAN iowOpened = FALSE;
USHORT iowProductId = 0;
VOID (*iowSetLedsFn)(ULONG leds) = NULL;
// IO Warrior file handle
IOWKIT_HANDLE iowHandle = NULL;

VOID
IowSetLeds40(ULONG val);

VOID
IowSetLeds24(ULONG val);


// Module header, includes version, description, and address of the module retriever function
winampVisHeader hdr = { VIS_HDRVER, "IO warrior visualization sample", getModule };

// VU meter (1 channel only?)
// Latency determines how often our module will be called
// so, for given latency of 25ms module will be called 1000ms/25ms times (40 times a second)
winampVisModule mod1 =
{
	"IO Warrior VU Meter",
	NULL,	// hwndParent
	NULL,	// hDllInstance
	0,		// sRate
	0,		// nCh
	25,		// latencyMS
	25,		// delayMS
	0,		// spectrumNch
	2,		// waveformNch
	{ 0, },	// spectrumData
	{ 0, },	// waveformData
	Config,
	Init,
	Render,
	Quit
};


// this is the only exported symbol. returns our main header.
// if you are compiling C++, the extern "C" { is necessary, so we just #ifdef it
#ifdef __cplusplus
extern "C" {
#endif
__declspec( dllexport ) winampVisHeader *winampVisGetHeader()
{
	return &hdr;
}
#ifdef __cplusplus
}
#endif

// getmodule routine from the main header. Returns NULL if an invalid module was requested,
// otherwise returns either mod1, mod2 or mod3 depending on 'which'.
winampVisModule *getModule(int which)
{
	switch (which)
	{
		case 0: return &mod1;
		default:return NULL;
	}
}

// configuration. Passed this_mod, as a "this" parameter. Allows you to make one configuration
// function that shares code for all your modules (you don't HAVE to use it though, you can make
// config1(), config2(), etc...)
void Config(struct winampVisModule *mod)
{
	MessageBox(mod->hwndParent, "IOW visualization module\n", "Configuration", MB_OK);
}

// Initialization. Registers our window class, creates our window, etc.
// returns 0 on success, 1 on failure.
int Init(struct winampVisModule *mod)
{
	int width = 256;
	int height = 32;

	//
	config_read(mod);

	// Register our window class
	WNDCLASS wc;

	memset(&wc,0,sizeof(wc));
	wc.lpfnWndProc = WndProc;				// our window procedure
	wc.hInstance = mod->hDllInstance;	// hInstance of DLL
	wc.lpszClassName = szAppName;			// our window class name
	if (!RegisterClass(&wc)) {
		MessageBox(mod->hwndParent,"Error registering window class","blah",MB_OK);
		return 1;
	}

	hMainWnd = CreateWindowEx(
		WS_EX_TOOLWINDOW|WS_EX_APPWINDOW,	// these exstyles put a nice small frame, 
							// but also a button in the taskbar
		szAppName,					// our window class name
		mod->description,				// use description for a window title
		WS_VISIBLE|WS_SYSMENU,				// make the window visible with a close button
		config_x,config_y,				// screen position (read from config)
		width,height,					// width & height of window (need to adjust client area later)
		mod->hwndParent,				// parent window (winamp main window)
		NULL,						// no menu
		mod->hDllInstance,				// hInstance of DLL
		0); // no window creation data

	if (!hMainWnd) 	{
		MessageBox(mod->hwndParent,"Error creating window","blah",MB_OK);
		return 1;
	}

	SetWindowLong(hMainWnd,GWL_USERDATA,(LONG)mod); // set our user data to a "this" pointer

	{	// adjust size of window to make the client area exactly width x height
		RECT r;
		GetClientRect(hMainWnd,&r);
		SetWindowPos(hMainWnd,0,0,0,width*2-r.right,height*2-r.bottom,SWP_NOMOVE|SWP_NOZORDER);
	}


	// create our doublebuffer
	memDC = CreateCompatibleDC(NULL);
	memBM = CreateCompatibleBitmap(memDC,width,height);
	oldBM = (HBITMAP) SelectObject(memDC,memBM);

	// OK, open IO warrior
	iowHandle = IowKitOpenDevice();
	// Check if IOW is opened
	if (iowHandle != NULL) {
		iowOpened = TRUE;
	}
	// Check for appropriate product ID
	iowProductId = IowKitGetProductId(iowHandle);
	switch (iowProductId) {
	case 0x1500:
		iowSetLedsFn = IowSetLeds40;
		break;
	case 0x1501:
		iowSetLedsFn = IowSetLeds24;
		break;
	default:
		IowKitCloseDevice(iowHandle);
		iowOpened = FALSE;
		iowHandle = NULL;
	}

	// show the window
	ShowWindow(hMainWnd,SW_SHOWNORMAL);

	return 0;
}

// IOW40 function
VOID
IowSetLeds40(ULONG val)
{
	IOWKIT_REPORT report;

	// Init report
	report.ReportID = 0;
	report.Value = val;
	// OK, write new value
	IowKitWrite(iowHandle, 0, (PCHAR) &report, sizeof(report));
}

// IOW24 function
VOID
IowSetLeds24(ULONG val)
{
	CHAR rep[3];

	// Init report
	rep[0] = 0;
	rep[1] = (CHAR) ((val >> 24) & 0xff);
	rep[2] = 0xff;
	// OK, write new value
	IowKitWrite(iowHandle, 0, &rep[0], sizeof(rep));
}

// Set leds
VOID
IowSetLeds(ULONG val)
{
	// Do nothing if IOW is't open
	if (iowOpened == FALSE) {
		return;
	}
	(*iowSetLedsFn)(val);
}

// Power on 1-8 LEDs, depending on signal
int Render(struct winampVisModule *mod)
{
	int x, y, leds1, leds2;

	// clear background
	Rectangle(memDC,0,0,256,32);
	// draw VU meter
	// One channel render only for now
	for (y = 0; y < 2; y ++)
	{
		int last = mod->waveformData[y][0];
		int total = 0;

		for (x = 1; x < 576; x++)
		{
			total += abs(last - mod->waveformData[y][x]);
			last = mod->waveformData[y][x];
		}
		// OK, we've got total signal
		// divide it to get 0-7 value
		total /= 288;
		if (total > 127) {
			total = 127;
		}
		if (total == 0) {
			total = 1;
		}
//#define		ONE_LED
// define ONE_LEN to see 8 bit one channel blinking
#ifdef		ONE_LED
		// Get set leds
		leds1 = total / 16;
		leds1 = (1 << leds1) - 1;
		leds1 = ~leds1;
		IowSetLeds(leds1 << 24);
#else
		if (y)
			leds2 = total / 32;
		else
			leds1 = total / 32;
#endif
		if (y) Rectangle(memDC,128,0,128+total,32);
		else Rectangle(memDC,128-total,0,128,32);
	}
	{ // copy doublebuffer to window
		HDC hdc = GetDC(hMainWnd);
		BitBlt(hdc,0,0,256,32,memDC,0,0,SRCCOPY);
		ReleaseDC(hMainWnd,hdc);
	}
#ifndef	ONE_LED
	leds1 = ~((1 << leds1) - 1) & 0xf;
	leds2 = ~((1 << leds2) - 1) & 0xf;
	// Merge leds
	leds1 = (leds2 << 4) | leds1;
	IowSetLeds(leds1 << 24);
#endif

	return 0;
}

// cleanup (opposite of init()). Destroys the window, unregisters the window class
void Quit(struct winampVisModule *mod)
{
	config_write(mod);		// write configuration
	SelectObject(memDC,oldBM);	// delete our doublebuffer
	DeleteObject(memDC);
	DeleteObject(memBM);	
	DestroyWindow(hMainWnd); // delete our window
	UnregisterClass(szAppName,mod->hDllInstance); // unregister window class
	if (iowOpened) {
		IowKitCloseDevice(iowHandle);
		iowHandle = NULL;
		iowOpened = FALSE;
	}
}


// window procedure for our window
LRESULT CALLBACK WndProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_CREATE:		return 0;
		case WM_ERASEBKGND: return 0;
		case WM_PAINT:
			{ // update from doublebuffer
				PAINTSTRUCT ps;
				RECT r;
				HDC hdc = BeginPaint(hwnd,&ps);
				GetClientRect(hwnd,&r);
				BitBlt(hdc,0,0,r.right,r.bottom,memDC,0,0,SRCCOPY);
				EndPaint(hwnd,&ps);
			}
		return 0;
		case WM_DESTROY: PostQuitMessage(0); return 0;
		case WM_KEYDOWN: // pass keyboard messages to main winamp window (for processing)
		case WM_KEYUP:
			{	// get this_mod from our window's user data
				winampVisModule *mod = (winampVisModule *) GetWindowLong(hwnd,GWL_USERDATA);
				PostMessage(mod->hwndParent,message,wParam,lParam);
			}
		return 0;
		case WM_MOVE:
			{	// get config_x and config_y for configuration
				RECT r;
				GetWindowRect(hMainWnd,&r);
				config_x = r.left;
				config_y = r.top;
			}
		return 0;
	}
	return DefWindowProc(hwnd,message,wParam,lParam);
}


void config_getinifn(struct winampVisModule *mod, char *ini_file)
{	// makes a .ini file in the winamp directory named "plugin.ini"
	char *p;

	GetModuleFileName(mod->hDllInstance, ini_file, MAX_PATH);
	p = ini_file+strlen(ini_file);
	while (p >= ini_file && *p != '\\')
		p--;
	if (++p >= ini_file)
		*p = 0;
	strcat(ini_file, "plugin.ini");
}


void config_read(struct winampVisModule *mod)
{
	char ini_file[MAX_PATH];
	int fix_config = 0;

	config_getinifn(mod,ini_file);
	config_x = GetPrivateProfileInt(mod->description,"Screen_x",config_x,ini_file);
	config_y = GetPrivateProfileInt(mod->description,"Screen_y",config_y,ini_file);
	// Check for safety
	if (config_x < 0) {
		config_x = 50;
		fix_config = 1;
	}
	if (config_y < 0) {
		config_y = 50;
		fix_config = 1;
	}
	// Check if values were crap and write new config then
	if (fix_config) {
		config_write(mod);
	}
}

void config_write(struct winampVisModule *mod)
{
	char string[32];
	char ini_file[MAX_PATH];

	config_getinifn(mod, ini_file);

	wsprintf(string,"%d", config_x);
	WritePrivateProfileString(mod->description, "Screen_x", string, ini_file);
	wsprintf(string,"%d", config_y);
	WritePrivateProfileString(mod->description, "Screen_y", string, ini_file);
}
