//
// openclos.cpp
// IO-Warrior SDK library
//
// IOW open & close functions
//
#include "stdafx.h"
#include "iowkit.h"
#include "iowdev.h"
#include "debug.h"

#pragma	pack(1)

// IOW handles & stuff
ULONG iowNumDevices = 0;
IOWKIT_DEVICE_INTERNAL iowHandles[IOWKIT_MAX_DEVICES];

// Here go internal functions
// Get device_interface_detail_data for given device
PSP_DEVICE_INTERFACE_DETAIL_DATA
IowKitGetDeviceInterfaceDetails(HDEVINFO devInfo, PSP_DEVICE_INTERFACE_DATA devData)
{
	PSP_DEVICE_INTERFACE_DETAIL_DATA devDetails;
	ULONG len, req_len;
	BOOL bOK;

	DbgOut(DBG_ALL, ("IowKitGetDeviceInterfaceDetails(): enter\n"));
	// Get length
    	bOK = SetupDiGetDeviceInterfaceDetail(devInfo, devData, NULL, 0, &len, NULL);
	// Fail if error
	if (len == 0) {
		DbgOut(DBG_ALL, ("GetDevIfDet(): not ok len %d\n", len));
		return NULL;
	}
	// Allocate device details
	devDetails = (PSP_DEVICE_INTERFACE_DETAIL_DATA) HeapAlloc(GetProcessHeap(), 0, len);
	// Fail if we couldn't alloc memory
	if (devDetails == NULL) {
		return NULL;
	}
	// Get details
	devDetails->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);
	bOK = SetupDiGetDeviceInterfaceDetail(devInfo,
						devData,
						devDetails,
						len,
						&req_len,
						NULL);
	// return result
	return devDetails;
}

HANDLE
IowKitGetDeviceHandleAsync(HDEVINFO devInfo, PSP_DEVICE_INTERFACE_DATA devData);
BOOLEAN
IowKitFireUpThread(PIOWKIT_DEVICE_INTERNAL iowHandle);

// Open device
HANDLE
IowKitGetDeviceHandle_(HDEVINFO devInfo, PSP_DEVICE_INTERFACE_DATA devData)
{
	HANDLE devHandle = INVALID_HANDLE_VALUE;
	PSP_DEVICE_INTERFACE_DETAIL_DATA devDetails;

	// Get device interface details
	devDetails = IowKitGetDeviceInterfaceDetails(devInfo, devData);
	// Try to open device
	if (devDetails) {
		DbgOut(DBG_ALL, ("IowKitGetDeviceHandle(): opening device %s\n", devDetails->DevicePath));
		devHandle = CreateFile(devDetails->DevicePath,
					GENERIC_READ | GENERIC_WRITE,
					FILE_SHARE_READ | FILE_SHARE_WRITE,
					NULL,
					OPEN_EXISTING,
					/* FILE_FLAG_OVERLAPPED */ 0,
					NULL);
		// Don't forget to free this!
		HeapFree(GetProcessHeap(), 0, devDetails);
	}

	return devHandle;
}

// ASYNC
// Open device in OVERLAPPED mode
HANDLE
IowKitGetDeviceHandleAsync(HDEVINFO devInfo, PSP_DEVICE_INTERFACE_DATA devData)
{
	HANDLE devHandle = INVALID_HANDLE_VALUE;
	PSP_DEVICE_INTERFACE_DETAIL_DATA devDetails;

	// Get device interface details
	devDetails = IowKitGetDeviceInterfaceDetails(devInfo, devData);
	// Try to open device
	if (devDetails) {
		DbgOut(DBG_ALL, ("IowKitGetDeviceHandle(): opening device %s\n", devDetails->DevicePath));
		devHandle = CreateFile(devDetails->DevicePath,
					GENERIC_READ | GENERIC_WRITE,
					FILE_SHARE_READ | FILE_SHARE_WRITE,
					NULL,
					OPEN_EXISTING,
					FILE_FLAG_OVERLAPPED,
					NULL);
		// Don't forget to free this!
		//free(devDetails);
		//delete devDetails;
		HeapFree(GetProcessHeap(), 0, devDetails);
	}

	return devHandle;
}

// Return TRUE if device represented by devHandle is IowKit device
// FALSE otherwise
BOOL
IowKitIsIowKitDevice(HANDLE devHandle, PUSHORT productId)
{
	HIDD_ATTRIBUTES hidAttr;
	BOOL bOK;

	// Set up structure for call
	hidAttr.Size = sizeof(hidAttr);
	// Get HID device attributes
	bOK = (*pHidD_GetAttributes)(devHandle, &hidAttr);
	// Fail if something's wrong
	// TODO: do something here?
	if (!bOK) {
		return FALSE;
	}
	// Compare vendorID & productID with IoWarrior's
	// NOTE: we don't care about VersionNumber yet
	bOK = ((hidAttr.VendorID == IOWKIT_VENDOR_ID) && ((hidAttr.ProductID == IOWKIT_PRODUCT_ID_IOW40) || (hidAttr.ProductID == IOWKIT_PRODUCT_ID_IOW24)));
	// Save Product ID
	*productId = hidAttr.ProductID;

	return bOK;
}

// Terminate thread and clean up after it
VOID
IowKitTerminateThread(PIOWKIT_DEVICE_INTERNAL iowDev)
{
	ULONG rc;

	// Check if thread has been started
	if (iowDev->ioThread == NULL) {
		return;
	}
	// Tell thread to terminate
	SetEvent(iowDev->endEvent);
	// Wait for thread to terminate
	rc = WaitForSingleObject(iowDev->threadTermEvent, INFINITE);
	// Wait for thread to terminate
	WaitForSingleObject(iowDev->ioThread, INFINITE);
	// Close thread handle
	CloseHandle(iowDev->ioThread);
	// Close all events & mutexes
	if (iowDev->readEvent) {
		CloseHandle(iowDev->readEvent);
	}
	if (iowDev->readEvent1) {
		CloseHandle(iowDev->readEvent1);
	}
	if (iowDev->endEvent) {
		CloseHandle(iowDev->endEvent);
	}
	if (iowDev->threadTermEvent) {
		CloseHandle(iowDev->threadTermEvent);
	}
	if (iowDev->bufMutex) {
		CloseHandle(iowDev->bufMutex);
	}
}

// Close IoWarrior device
VOID
IowKitCloseOneDevice(PIOWKIT_DEVICE_INTERNAL iowDev)
{
	ULONG i;

	// Terminate async I/O thread if simple interface was open
	if (iowDev->isLegacy && (iowDev->openMode == IOW_OPEN_SIMPLE)) {
		IowKitTerminateThread(iowDev);
	}
	// Close all handles
	for (i = 0; i < iowDev->numPipes; i++) {
		CloseHandle(iowDev->iowHandle[i]);
	}
	// Poison cookie
	iowDev->cookie = 0xf00fbeef;
}

VOID IOWKIT_API
IowKitCloseDevice(IOWKIT_HANDLE devHandle)
{
	PIOWKIT_DEVICE_INTERNAL iowDev;
	ULONG i;

	iowDev = (PIOWKIT_DEVICE_INTERNAL) devHandle;
	// Check IOW handle
	if (iowDev == NULL || (iowDev->cookie != IOWKIT_DEVICE_COOKIE)) {
		return;
	}
	// Fail if this is not main device
	if (!iowDev->isMain) {
		return;
	}
	// Close all devices
	for (i = 0; i < iowNumDevices; i++) {
		IowKitCloseOneDevice(&iowHandles[i]);
	}
	// OK, we closed IOWs
	iowNumDevices = 0;
	// Unload HID library
	IowKitUnloadHid();
	// Disable DBG
	//IowKitDisableDbg();
}

// Get IOW serial number
BOOLEAN
IowKitGetSerial(HANDLE pipeHandle, PWCHAR buffer, ULONG bufLen)
{
	BOOLEAN rc;

	// We don't support S/N yet
	rc = (*pHidD_GetSerialNumberString)(pipeHandle, buffer, bufLen);
	buffer[8] = 0;
	// Check if S/N is empty
	if (wcslen(buffer) != 8) {

		return FALSE;
	}

	return rc;
}

// Find handle by IOW device number or serial number
PIOWKIT_DEVICE_INTERNAL
IowKitFindByNumberOrSerial(ULONG iowNum, PWCHAR serialNumber, BOOLEAN alloc)
{
	ULONG i;

	// Find by IOW device number if S/N is empty
	if (serialNumber == NULL) {
		// Check if number is too high
		// and fail in that case
		if (iowNum > (iowNumDevices - 1)) {

			return NULL;
		}

		// Return IOW handle
		return &iowHandles[iowNum];
	} else {
		// Find IOW by serial number, allocate new slot
		// if S/N is not found
		for (i = 0; i < iowNumDevices; i++) {
			// OK, we found S/N, return handle
			if (!wcscmp(iowHandles[i].serialNumber, serialNumber)) {

				return &iowHandles[i];
			}
		}
		// Fail if there are already 16 devices open
		// or if we don't need to alloc new slot
		if ((iowNumDevices == IOWKIT_MAX_DEVICES) || (!alloc)) {
			return NULL;
		}
		// Return next free slot
		iowHandles[iowNumDevices].refCount = 0;
		iowHandles[iowNumDevices].numPipes = 0;
		// Copy serial number
		memcpy(iowHandles[iowNumDevices].serialNumber, serialNumber, sizeof(iowHandles[iowNumDevices].serialNumber));
		iowNumDevices++;

		return &iowHandles[iowNumDevices-1];
	}
}

BOOLEAN
IowiKitGetReportLength(HANDLE devHandle, PULONG pLength)
{
	PHIDP_PREPARSED_DATA hidData;
	HIDP_CAPS hidCaps;
	NTSTATUS status;

	// Zero size
	*pLength = 0;
	// Get preparsed data
	if (!(*pHidD_GetPreparsedData)(devHandle, &hidData)) {

		// Fail if we can't
		return FALSE;
	}
	// Get HID caps
	status = (*pHidP_GetCaps)(hidData, &hidCaps);
	// Free preparsed data
	(*pHidD_FreePreparsedData)(hidData);
	// Fail if error
	if (status != HIDP_STATUS_SUCCESS) {
		return FALSE;
	}
	// Save report size
	*pLength = hidCaps.InputReportByteLength;

	return TRUE;
}

// Enumerate IOW devices
BOOLEAN
IowiKitEnumDevices(PHANDLE iowOldDevs, PULONG pNumDev, PULONG pNumOld)
{
	HANDLE devHandle;
	GUID hidGuid;
	HDEVINFO devInfo;
	SP_INTERFACE_DEVICE_DATA devData;
	BOOL bEnum = TRUE, bFound = FALSE;
	WCHAR serialNumber[IOWKIT_SN_LEN + 1];
	PIOWKIT_DEVICE_INTERNAL iowDev;
	ULONG i, j;
	USHORT pid;
	ULONG numDev, numOld;

	// Init everything
	*pNumDev = *pNumOld = 0;
	numDev = numOld = 0;
	memset(iowHandles, 0, sizeof(iowHandles));
	// Load HID library
	if (!IowKitLoadHid()) {
		return FALSE;
	}
	// Get HID GUID
	(*pHidD_GetHidGuid)(&hidGuid);
	// Start HID device enumeration
	devInfo = SetupDiGetClassDevs(&hidGuid, NULL, NULL, DIGCF_PRESENT | DIGCF_DEVICEINTERFACE);
	// Check if error
	if (devInfo == INVALID_HANDLE_VALUE) {
		return FALSE;
	}
	// Set up DEVICE_DATA for enumeration
	devData.cbSize = sizeof(devData);
	// Init
	i = -1;
	// Enumerate IOW devices
	while (bEnum) {
		i++;
		// Enumerate device
		bEnum = SetupDiEnumDeviceInterfaces(devInfo,
							NULL,
							&hidGuid,
							i,
							&devData);
		// Exit enumeration loop
		// Though specs say we should stop only when
		// bEnum == FALSE and GetLastError() == ERROR_NO_MORE_ITEMS
		// we exit anyway?
		// TODO: FIX IT?
		if (!bEnum) {
			break;
		}
		// Open device
		devHandle = IowKitGetDeviceHandleAsync(devInfo, &devData);
		// Check if we have handle
		if (devHandle == INVALID_HANDLE_VALUE) {
			continue;
		}
		// OK, we opened device, check if it's not IO-Warrior
		// and continue loop in such case
		if (!IowKitIsIowKitDevice(devHandle, &pid)) {
			// Close handle and find next device
			CloseHandle(devHandle);
			continue;
		}
		// We found IOW
		bFound = TRUE;
		// We've got IO-Warrior device handle
		// now check it's S/N and stuff like that
		// XXX rewrite GetSer to indicate FAILURE/no SERIAL NUMBER cases???
		// If device doesn't have serial handle gets saved
		// in iowOldDevs[] array
		if (IowKitGetSerial(devHandle, serialNumber, sizeof(serialNumber))) {
			//DbgOut(DBG_ALL, ("Device got serial %s\n", serialNumber));
			// OK, we've got IO-Warrior with serial number
			// Get its slot and save handle there
			iowDev = IowKitFindByNumberOrSerial(0, serialNumber, TRUE);
			// Fail if there are no free slots
			if (iowDev == NULL) {
				// Close device
				CloseHandle(devHandle);
				// Destroy devInfo
				SetupDiDestroyDeviceInfoList(devInfo);
				// Close everything
				for (i = 0; i < numDev; i++) {
					for (j = 0; j < iowHandles[i].numPipes; j++) {
						CloseHandle(iowHandles[i].iowHandle[j]);
					}
				}
				// Zero iowHandles
				memset(iowHandles, 0, sizeof(iowHandles));
				iowNumDevices = 0;
				// Close legacy IOW handles
				for (i = 0; i < numOld; i++) {
					CloseHandle(iowOldDevs[i]);
				}

				// We failed
				return FALSE;
			}
			// Save pipe handle for later use
			// We will sort pipes later
			iowDev->iowHandle[iowDev->numPipes] = devHandle;
			iowDev->numPipes++;
			// Save IOW Product ID
			iowDev->productId = pid;
			// Open mode is zero for new IOW devices!
			iowDev->openMode = 0;
			// Set timeout
			iowDev->ioTimeout = INFINITE;
		} else {
			// Device doesn't have serial number
			// Save "old" device handle in array
			iowOldDevs[numOld++] = devHandle;
			DbgOut(DBG_ALL, ("One more old device\n"));
		}
	}
	// Destroy devInfo
	SetupDiDestroyDeviceInfoList(devInfo);
	// Fail if we didn't find any IOWs
	if (!bFound) {
		return FALSE;
	}
	*pNumOld = numOld;
	*pNumDev = iowNumDevices;

	return TRUE;
}

IOWKIT_HANDLE IOWKIT_API
IowKitOpenDevice(VOID)
{
	HANDLE iowOldDevs[IOWKIT_MAX_DEVICES * IOWKIT_MAX_PIPES];
	IOWKIT_DEVICE_INTERNAL iowDevs[IOWKIT_MAX_DEVICES];
	HANDLE temp;
	ULONG numOld, numDev;
	ULONG repLen;
	ULONG i, j, k;

	// Init everything
	memset(iowHandles, 0, sizeof(iowHandles));
	memset(iowOldDevs, 0, sizeof(iowOldDevs));
	memset(iowDevs, 0, sizeof(iowDevs));
	// 1. Enumerate IOWs
	if (!IowiKitEnumDevices(iowOldDevs, &numDev, &numOld)) {
		// Enumeration failed, we can't open IOWs
		return NULL;
	}
	// OK, we've got old & new IOWs in arrays
	// 2. Special-case - 1 old IOW
	if (numOld == 2 && (iowLegacyOpenMode == IOW_OPEN_SIMPLE)) {
		// Fail if we can't get report length
		if (!IowiKitGetReportLength(iowOldDevs[0], &repLen)) {
			goto failed;
		}
		// Set up IOW handle
		iowDevs[0].productId = IOWKIT_PRODUCT_ID_IOW40;
		iowDevs[0].numPipes = 2;
		iowDevs[0].refCount = 1;
		iowDevs[0].isLegacy = TRUE;
		iowDevs[0].openMode = IOW_OPEN_SIMPLE;
		// Set timeout
		iowDevs[0].ioTimeout = INFINITE;
		// Save endpipe handles
		if (repLen == 5) {
			iowDevs[0].iowHandle[0] = iowOldDevs[0];
			iowDevs[0].iowHandle[1] = iowOldDevs[1];
		} else {
			iowDevs[0].iowHandle[0] = iowOldDevs[1];
			iowDevs[0].iowHandle[1] = iowOldDevs[0];
		}
		// Invalidate old device handles
		iowOldDevs[0] = INVALID_HANDLE_VALUE;
		iowOldDevs[1] = INVALID_HANDLE_VALUE;
		// There is one old device
		numOld = 1;
		goto skip_old;
	}
	// 3. Close either simple or complex endpipes for old IOWs
	for (i = 0, j = 0; i < numOld; i++) {
		// Get report length
		if (!IowiKitGetReportLength(iowOldDevs[i], &repLen)) {
			goto failed;
		}
		// Set up IOW handle
		iowDevs[j].productId = IOWKIT_PRODUCT_ID_IOW40;
		iowDevs[j].numPipes = 1;
		iowDevs[j].refCount = 1;
		iowDevs[j].isLegacy = TRUE;
		// Set timeout
		iowDevs[j].ioTimeout = INFINITE;
		// Check if it's primary interface and save handle
		// in such case
		switch (repLen) {
		// Primary device interface
		case 5:
		// Check legacyOpenMode and save IOW handle if appropriate
		if (iowLegacyOpenMode == IOW_OPEN_SIMPLE) {
			// Set up IOW handle
			iowDevs[j].openMode = IOW_OPEN_SIMPLE;
			// Save primary interface handle
			iowDevs[j].iowHandle[0] = iowOldDevs[i];
			j++;
		} else {
			// Close this interface
			CloseHandle(iowOldDevs[i]);
		}
		// Invalidate handle so we don't close it twice
		iowOldDevs[i] = INVALID_HANDLE_VALUE;
		break;
		// Secondary interface
		case 8:
		// Check legacyOpenMode and save IOW handle if appropriate
		if (iowLegacyOpenMode == IOW_OPEN_COMPLEX) {
			// Set up IOW handle
			iowDevs[j].openMode = IOW_OPEN_COMPLEX;
			// Save primary interface handle
			iowDevs[j].iowHandle[0] = iowOldDevs[i];
			j++;
		} else {
			// Close this interface
			CloseHandle(iowOldDevs[i]);
		}
		// Invalidate handle so we don't close it twice
		iowOldDevs[i] = INVALID_HANDLE_VALUE;
		break;
		}
	}
	// Save real numOld value
	numOld = j;
skip_old:
	// 4. Exchange new IOW endpipe handles
	for (i = 0; i < numDev; i++) {
		// Fail if we can't get report length
		if (!IowiKitGetReportLength(iowHandles[i].iowHandle[0], &repLen)) {
			goto failed;
		}
		switch (iowHandles[i].productId) {
		case IOWKIT_PRODUCT_ID_IOW40:
		// Pipes coming in wrong order, exchange them
		if (repLen != 5) {
			temp = iowHandles[i].iowHandle[0];
			iowHandles[i].iowHandle[0] = iowHandles[i].iowHandle[1];
			iowHandles[i].iowHandle[1] = temp;
		}
		break;
		case IOWKIT_PRODUCT_ID_IOW24:
		// Pipes coming in wrong order, exchange them
		if (repLen != 3) {
			temp = iowHandles[i].iowHandle[0];
			iowHandles[i].iowHandle[0] = iowHandles[i].iowHandle[1];
			iowHandles[i].iowHandle[1] = temp;
		}
		break;
		// We don't support other IOW PIDs!
		default:
			goto failed;
		}
	}
	// 5. Move "old" IOWs to the beginning of array, and new to the end
	// New IOWs to the end of array
	if (numOld) {
		for (i = 0; i < numDev; i++) {
			iowHandles[i+numOld] = iowHandles[i];
		}
		for (i = 0; i < numOld; i++) {
			iowHandles[i] = iowDevs[i];
		}
//		memcpy(iowHandles, iowHandles + numOld, numDev * sizeof(IOWKIT_DEVICE_INTERNAL));
		// Old IOWs to the beginning of array
//		memcpy(iowDevs, iowHandles, numOld * sizeof(IOWKIT_DEVICE_INTERNAL));
	}
	// numDev is real number of IOWs now
	numDev += numOld;
	numOld = 0;
	// 1st IOW is main handle
	iowHandles[0].isMain = TRUE;
	// Now fire up async I/O threads and set up final bits
	for (i = 0; i < numDev; i++) {
		if (i > 0) {
			iowHandles[i].isMain = FALSE;
		}
		// Set cookie
		iowHandles[i].cookie = IOWKIT_DEVICE_COOKIE;
		iowHandles[i].ioThread = NULL;
		// Start async I/O thread for this IOW if it's primary
		// interface
		if (iowHandles[i].openMode == IOW_OPEN_SIMPLE) {
		if (!IowKitFireUpThread(&iowHandles[i])) {
			// Clean up if failed
			for (k = 0; k < i; k++) {
				// Terminate thread only if it has been started
				// Note: devices are still open
				if (iowHandles[k].openMode == IOW_OPEN_SIMPLE) {
					IowKitTerminateThread(&iowHandles[i]);
				}
			}
			// Clean up after failure
			goto failed;
		}
		} else
		if (iowHandles[i].openMode == IOW_OPEN_COMPLEX) {
			(*pHidD_SetNumInputBuffers)(iowHandles[i].iowHandle[0], 44);
		}
		// Check and set number of buffers for second interface
		if (iowHandles[i].numPipes > 1) {
			(*pHidD_SetNumInputBuffers)(iowHandles[i].iowHandle[1], 44);
			// NOTE, we don't check number of buffers yet
		}
	}
	// Set up num devices and return IOW handle
	iowNumDevices = numDev;

	return iowHandles;
failed:
	// Failure handling
	// Close all IOWs in iowHandles[] array
	for (i = 0; i < numDev; i++) {
		IowKitCloseOneDevice(&iowHandles[i]);
	}
	// Close all legacy IOWs if any
	for (i = 0; i < numOld; i++) {
		CloseHandle(iowOldDevs[i]);
	}
	// Zero everything
	memset(iowHandles, 0, sizeof(iowHandles));
	iowNumDevices = 0;

	// We failed to open IOWs
	return NULL;
}
