unit IOIRMain;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, Buttons, StdCtrls, ComCtrls,
  JvHidControllerClass;

const
  cCodeMercenariesVID = $07C0;
  cIOWarriorPID1      = $1500;
  cIOWarriorPID2      = $1501;
  cIRCommand          = $0C;
  cIRReportID         = $0C;
  cIROff              = $00;
  cIROn               = $01;

  // translates RC5-Adress to device description
  AddressNames: array [0..31] of PChar =
   (
    'TV1',
    'TV2',
    'Teletext',
    'Video',
    'LV1',
    'VCR1',
    'VCR2',
    'Experimental',
    'Sat1',
    'Camera',
    'Sat2',
    '11',
    'CDV',
    'Camcorder',
    '14',
    '15',
    'Pre-amp1',
    'Tuner',
    'Recorder1',
    'Pre-amp2',
    'CD Player',
    'Phono',
    'SatA',
    'Recorder2',
    '24',
    '25',
    'CDR',
    '27',
    '28',
    'Lighting1',
    'Lighting2',
    'Phone'
   );

type
  // IR record read from IO-Warrior 24
  PIOWarriorIRInputReport = ^TIOWarriorIRInputReport;
  TIOWarriorIRInputReport = packed record
    Command: Byte;  // Command 0..63 of IR RC5 code
    Address: Byte;  // Device ID 0..31 of IR RC5 code
    Empty: array [1..5] of Byte;  // 5 unused bytes containing 0
  end;

  // IR command record for IO-Warrior 24
  PIOWarriorIROutputReport = ^TIOWarriorIROutputReport;
  TIOWarriorIROutputReport = packed record
    ReportID: Byte;  // always part of the write record
    IOData: array [0..6] of Byte;  // 7 bytes for IO-Warrior
  end;

  // RC5 data to keys assignment data structure
  PKeyData = ^TKeyData;
  TKeyData = record
    Command: Byte;        // decoded RC5 command
    Address: Byte;        // decoded RC5 address
    Toggle: Boolean;      // toggles with any key press on IR remote
    Key: Word;            // assigned key
    Shift: TShiftState;   // assigned Ctrl/Alt/Shift
    ExtendedKey: Boolean; // numpad bit for key
  end;

  TMainForm = class(TForm)
    HidCtl: TJvHidDeviceController;
    IOWarriorDetected: TLabel;
    CodeList: TListView;
    Description: TLabel;
    Clear: TButton;
    procedure HidCtlDeviceChange(Sender: TObject);
    procedure HidCtlDeviceData(HidDev: TJvHidDevice;
      ReportID: Byte; const Data: Pointer; Size: Word);
    procedure CodeListDblClick(Sender: TObject);
    procedure ClearClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  public
    IOWarrior: TJvHidDevice;
    KeyData: TKeyData;
    LastKeyData: TKeyData;
    RepeatCount: Integer;
    procedure SendKeycodes(KeyDataPtr: PKeyData);
    procedure NewCode(const KeyData: TKeyData; Description: string);
  end;

var
  MainForm: TMainForm;

implementation

uses
  IniFiles,
  KeyEdit;

{$R *.dfm}

const
  cAssignmentsSection = 'Assignments';
  cIniFileName = 'IOIRWarrior.ini';

// separate RC5 data

procedure DecodeRC5Data(const Report: TIOWarriorIRInputReport;
  var KeyData: TKeyData);
begin
  // 6 bits used
  KeyData.Command := Report.Command and $3F;
  // Bit 6 of Command is stored inverted in Address
  KeyData.Command := KeyData.Command or
    ((not Report.Address) and $40);
  // 5 bits used
  KeyData.Address := Report.Address and $1F;
  // the toggle bit changes each time a different key
  // is pressed on the IR remote
  KeyData.Toggle  := (Report.Address and $20) <> 0;
end;

// send the key combination to Windows
// as if originating from the keyboard

procedure TMainForm.SendKeycodes(KeyDataPtr: PKeyData);
var
  ExtKey: DWORD;
begin
  // suppress first repeat of key press from remote
  KeyData := KeyDataPtr^;
  if KeyData.Toggle = LastKeyData.Toggle then
    Inc(RepeatCount)
  else
    RepeatCount := 0;
  if (KeyData.Toggle = LastKeyData.Toggle) and (RepeatCount = 1) then
    Exit;
  LastKeyData := KeyData;

  // no assignment so do not send anything
  if KeyData.Key = 0 then
    Exit;

  // key down for Shift, Alt, Ctrl
  if ssShift in KeyData.Shift then
    keybd_event(VK_SHIFT, 0, 0, 0);
  if ssCtrl in KeyData.Shift then
    keybd_event(VK_CONTROL, 0, 0, 0);
  if ssAlt in KeyData.Shift then
    keybd_event(VK_MENU, 0, 0, 0);

  // add extended key bit
  if KeyData.ExtendedKey then
    ExtKey := KEYEVENTF_EXTENDEDKEY
  else
    ExtKey := 0;
  // send key down then key up
  keybd_event(KeyData.Key, 0, ExtKey, 0);
  keybd_event(KeyData.Key, 0, ExtKey or KEYEVENTF_KEYUP, 0);

  // key up for Shift, Alt, Ctrl (reversed sequence)
  if ssAlt in KeyData.Shift then
    keybd_event(VK_MENU, 0, KEYEVENTF_KEYUP, 0);
  if ssCtrl in KeyData.Shift then
    keybd_event(VK_CONTROL, 0, KEYEVENTF_KEYUP, 0);
  if ssShift in KeyData.Shift then
    keybd_event(VK_SHIFT, 0, KEYEVENTF_KEYUP, 0);
end;

procedure TMainForm.NewCode(const KeyData: TKeyData;
  Description: string);
var
  PData: PKeyData;
  Item: TListItem;
begin
  // fill in a new list entry
  Item := CodeList.Items.Add;
  Item.Caption := AddressNames[KeyData.Address];
  Item.SubItems.Add(Format('%d', [KeyData.Command]));
  Item.SubItems.Add(Description);
  Item.SubItems.Add(KeyName(KeyData.Key, KeyData.ExtendedKey));
  // allocate a TKeyData record, fill it and assign it to the list entry
  GetMem(PData, SizeOf(TKeyData));
  PData^ := KeyData;
  Item.Data := PData;
end;

procedure TMainForm.FormCreate(Sender: TObject);
var
  I: Integer;
  S: string;
  Ini: TIniFile;
  List: TStringList;
  KeyData: TKeyData;
  Description: string;
begin
  // load saved settings
  List := TStringList.Create;
  Ini := TIniFile.Create(ExtractFilePath(ParamStr(0)) +
    cIniFileName);
  Ini.ReadSectionValues(cAssignmentsSection, List);
  for I := 0 to List.Count-1 do
  begin
    S := List.Names[I];
    KeyData.Command := StrToInt(S) shr 8;
    KeyData.Address := StrToInt(S) and $FF;
    KeyData.Toggle := False;
    S := List.Values[S];
    KeyData.Key :=
      StrToInt(Copy(S, 0, 3));
    KeyData.Shift := [];
    if S[5] = '1' then
      KeyData.Shift := KeyData.Shift + [ssShift];
    if S[7] = '1' then
      KeyData.Shift := KeyData.Shift + [ssAlt];
    if S[9] = '1' then
      KeyData.Shift := KeyData.Shift + [ssCtrl];
    KeyData.ExtendedKey := (S[11] = '1');
    Description := Copy(S, 13, Length(S));
    NewCode(KeyData, Description);
  end;
  Ini.Free;
  List.Free;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
var
  I: Integer;
  Ini: TIniFile;
  Ident: string;
  Value: string;
  KeyDataPtr: PKeyData;
begin
  // save the assignments
  Ini := TIniFile.Create(ExtractFilePath(ParamStr(0)) +
    cIniFileName);
  Ini.EraseSection(cAssignmentsSection);
  for I := 0 to CodeList.Items.Count-1 do
  begin
    KeyDataPtr := CodeList.Items[I].Data;
    Ident := Format('%u',
      [(KeyDataPtr^.Command shl 8) + KeyDataPtr^.Address]);
    Value := Format('%.3u %u %u %u %u %s',
      [KeyDataPtr^.Key,
       Ord(ssShift in KeyDataPtr^.Shift),
       Ord(ssAlt in KeyDataPtr^.Shift),
       Ord(ssCtrl in KeyDataPtr^.Shift),
       Ord(KeyDataPtr^.ExtendedKey),
       CodeList.Items[I].SubItems[1]]);
    Ini.WriteString(cAssignmentsSection, Ident, Value);
  end;
  Ini.Free;

  // cleanup like any wellbehaved program should do
  ClearClick(Self);
end;

// Callback to identify correct IO-Warrior 24 interface

function FindIOWarrior(HidDev: TJvHidDevice): Boolean; stdcall;
begin
  // each IOWarrior shows as two devices
  // we need access to the one with InputReportByteLength = 8
  // the other device is for access to the IO pins
  Result :=
    (HidDev.Attributes.VendorID = cCodeMercenariesVID) and
    (HidDev.Attributes.ProductID = cIOWarriorPID2) and
    (HidDev.Caps.InputReportByteLength = 8);
end;

// check for plug or unplug of an IO-Warrior 24

procedure TMainForm.HidCtlDeviceChange(Sender: TObject);
const
  cIOWMessage = 'IO-Warrior with IR found';
var
  BytesWritten: Cardinal;
  IOWarriorOutputReport: TIOWarriorIROutputReport;
begin
  // free the device object if the device has been unplugged
  if Assigned(IOWarrior) and not IOWarrior.IsPluggedIn then
  begin
    IOWarrior.Free;
    IOWarrior := nil;
  end;

  // if no IOWarrior in use then search for one
  if not Assigned(IOWarrior) then
    if HidCtl.CheckOutByCallback(IOWarrior, FindIOWarrior) then
    begin
      // found, initialize IR functions
      FillChar(IOWarriorOutputReport,
        SizeOf(IOWarriorOutputReport), #0);
      IOWarriorOutputReport.ReportID  := cIRCommand;
      IOWarriorOutputReport.IOData[0] := cIROn;
      IOWarrior.WriteFile(IOWarriorOutputReport,
        SizeOf(IOWarriorOutputReport), BytesWritten);
    end;

  Description.Enabled := Assigned(IOWarrior);
  if Assigned(IOWarrior) then
    IOWarriorDetected.Caption := cIOWMessage
  else
    IOWarriorDetected.Caption := 'No ' + cIOWMessage;
end;

// an IR command was received

procedure TMainForm.HidCtlDeviceData(HidDev: TJvHidDevice;
  ReportID: Byte; const Data: Pointer; Size: Word);
var
  I: Integer;
  KeyData: TKeyData;
  PData: PKeyData;
  FoundAt: Integer;
  Item: TListItem;
  IOWarriorInputReport: TIOWarriorIRInputReport;
begin
  IOWarriorInputReport := PIOWarriorIRInputReport(Data)^;
  if ReportID = cIRReportID then
  begin
    // extract RC5 data
    FillChar(KeyData, SizeOf(KeyData), #0);
    DecodeRC5Data(IOWarriorInputReport, KeyData);

    // search the list of key assignments
    FoundAt := -1;
    for I := 0 to CodeList.Items.Count-1 do
    begin
      PData := PKeyData(CodeList.Items[I].Data);
      if (KeyData.Command = PData^.Command) and
        (KeyData.Address = PData^.Address) then
      begin
        FoundAt := I;
        Break;
      end;
    end;

    if FoundAt = -1 then
      // add new RC5 key to list
      NewCode(KeyData, '--')
    else
    begin
      // RC5 key already in list, select it
      Item := CodeList.Items[FoundAt];
      CodeList.Selected := Item;
      // send assigned key combination
      SendKeycodes(Item.Data);
    end;
  end;
end;

// edit list entry

procedure TMainForm.CodeListDblClick(Sender: TObject);
var
  Item: TListItem;
  PData: PKeyData;
begin
  // check for a selected list entry
  Item := CodeList.Selected;
  if Assigned(Item) then
    // create edit dialog
    with TKeyEditForm.Create(Self) do
    begin
      // fill dialog with values
      PData := Item.Data;
      KeyData := PData^;
      Address.Caption := Item.Caption;
      Value.Caption   := Item.SubItems[0];
      Name.Text       := Item.SubItems[1];
      Keys.Text       := Item.SubItems[2];
      ExtKey.Checked  := KeyData.ExtendedKey;
      if ShowModal = mrOk then
      begin
         // retrieve edited values
         Item.SubItems[1] := Name.Text;
         Item.SubItems[2] := Keys.Text;
         PData^ := KeyData;
      end;
      Free;
    end;
end;

// delete all key assignments

procedure TMainForm.ClearClick(Sender: TObject);
var
  I: Integer;
begin
  CodeList.Items.BeginUpdate;
  for I := CodeList.Items.Count-1 downto 0 do
  begin
    FreeMem(CodeList.Items[I].Data);
    CodeList.Items.Delete(I);
  end;
  CodeList.Items.EndUpdate;
end;

end.
