//
// readwrit.cpp
//
// IO-Warrior SDK library
// Read & write functions
//
#include "stdafx.h"
#include "iowkit.h"
#include "iowdev.h"
#include "debug.h"

#pragma	pack(1)

// Internal IOW-Kit functions
// Get numPipe handle
HANDLE
IowKitGetPipeHandle(IOWKIT_HANDLE devHandle, ULONG numPipe)
{
	PIOWKIT_DEVICE_INTERNAL iowDev = (PIOWKIT_DEVICE_INTERNAL) devHandle;
	HANDLE handle = INVALID_HANDLE_VALUE;

	// Check devHandle
	if (iowDev == NULL || (iowDev->cookie != IOWKIT_DEVICE_COOKIE)) {
		return handle;
	}
	// Check number of pipes
	// Note, pipes go from zero to numPipe-1
	if ((iowDev->numPipes - 1) < numPipe) {
		return handle;
	}
	// Get handle
	handle = iowDev->iowHandle[numPipe];

	return handle;
}

// Dummy read completion function
VOID __stdcall
IowDummyCompletion(DWORD dummy1, DWORD dummy2, OVERLAPPED *dummy3)
{
}

// Set I/O timeout for given IOW device
BOOLEAN IOWKIT_API
IowKitSetTimeout(IOWKIT_HANDLE devHandle, ULONG timeout)
{
	PIOWKIT_DEVICE_INTERNAL iowDev;

	// Get iowDev
	iowDev = (PIOWKIT_DEVICE_INTERNAL) devHandle;
	// Check for bogus handle
	if (iowDev == NULL) {

		return FALSE;
	}
	__try {
	// Check cookie
	if (iowDev->cookie != IOWKIT_DEVICE_COOKIE) {
		DbgOut(DBG_ALL, ("Bad cookie?\n"));

		return FALSE;
	}
	// Save new timeout value
	iowDev->ioTimeout = timeout;

	return TRUE;
	}
	__except ( EXCEPTION_EXECUTE_HANDLER ) {
		DbgOut(DBG_ALL, ("Bad handle, %p\n", iowDev));

		return FALSE;
	}
}

// Set I/O timeout for given IOW device
BOOLEAN IOWKIT_API
IowKitCancelIo(IOWKIT_HANDLE devHandle, ULONG numPipe)
{
	PIOWKIT_DEVICE_INTERNAL iowDev;
	HANDLE pipeHandle;

	// Get iowDev
	iowDev = (PIOWKIT_DEVICE_INTERNAL) devHandle;
	// Check for bogus handle
	if (iowDev == NULL) {

		return FALSE;
	}
	__try {
	// Check cookie
	if (iowDev->cookie != IOWKIT_DEVICE_COOKIE) {
		DbgOut(DBG_ALL, ("Bad cookie?\n"));

		return FALSE;
	}
	// Get pipe handle
	pipeHandle = IowKitGetPipeHandle(devHandle, numPipe);
	// Pipe doesn't exist, fail
	if (pipeHandle == INVALID_HANDLE_VALUE) {

		return FALSE;
	}
	// Cancel pending I/O for given pipe
	CancelIo(pipeHandle);
	// TODO: Not complete yet! Deal with zero pipe and stuff
	// Add some event for synchronization

	return TRUE;
	}
	__except ( EXCEPTION_EXECUTE_HANDLER ) {
		DbgOut(DBG_ALL, ("Bad handle, %p\n", iowDev));

		return FALSE;
	}
}

// Low-level read-write stuff
// IowKitWrite(): write 32 bit DWORD value to device
ULONG IOWKIT_API 
IowKitWrite(IOWKIT_HANDLE devHandle, ULONG numPipe,
		PCHAR buffer, ULONG length)
{
	PIOWKIT_DEVICE_INTERNAL iowDev;
	HANDLE iowHandle;
	ULONG rc, i;
	OVERLAPPED ovlp;
	BOOL bOK;

	// Get iowDev
	iowDev = (PIOWKIT_DEVICE_INTERNAL) devHandle;
	// Check for bogus handle
	if (iowDev == NULL) {

		return 0;
	}
	__try {
	// Check cookie
	if (iowDev->cookie != IOWKIT_DEVICE_COOKIE) {
		DbgOut(DBG_ALL, ("Bad cookie?\n"));

		return 0;
	}
	}
	__except ( EXCEPTION_EXECUTE_HANDLER ) {
		DbgOut(DBG_ALL, ("Bad handle, %p\n", iowDev));

		return 0;
	}
	iowHandle = IowKitGetPipeHandle(iowDev, numPipe);
	// Check if pipe is valid
	if (iowHandle == INVALID_HANDLE_VALUE) {
		SetLastError(ERROR_INVALID_PARAMETER);

		return 0;
	}
	// Zero overlapped structure
	ZeroMemory(&ovlp, sizeof(ovlp));
	DbgOutF(DBG_ALL, (dbgf, "Writing %d %x %d [", numPipe, buffer, length));
	for (i = 0; i < length; i++) {
		DbgOutF(DBG_ALL, (dbgf, "%x ", (ULONG)(UCHAR) buffer[i]));
	}
	DbgOutF(DBG_ALL, (dbgf, "]\n"));
	// Perform async write waiting for results
	bOK = WriteFileEx(iowHandle, buffer, length, &ovlp, &IowDummyCompletion);
	// Check for error
	if (!bOK && (GetLastError() != ERROR_IO_PENDING)) {
		DbgOut(DBG_ALL, ("IowKitWrite(): failed to write, error %d\n", GetLastError()));

		return 0;
	}
	// Sleep
	// NOTE: We don't use timeout here!
	rc = SleepEx(INFINITE, TRUE);
	// TODO: perform more strict checks
	if (rc == WAIT_IO_COMPLETION) {
		// Return number of bytes written
		return length;
	}
	// We failed with timeout, set error
	SetLastError(ERROR_TIMEOUT);

	// Error otherwise
	return 0;
}

BOOLEAN
IowKitInsertBuffer(PIOWKIT_DEVICE_INTERNAL iowDev, DWORD value)
{
	BOOLEAN bOk = TRUE;

	// Check if buffer is overflowing
	if (iowDev->numBuf == IOW_BUFFER_SIZE) {
		bOk = FALSE;
		// Loose 1st value in buffer
		if (iowDev->bufStart == iowDev->bufEnd) {
			iowDev->bufStart = IOWKIT_BUFFER_POS(iowDev->bufStart + 1);
		} else {
			iowDev->bufStart = IOWKIT_BUFFER_POS(iowDev->bufStart + 1);
			iowDev->bufEnd = IOWKIT_BUFFER_POS(iowDev->bufEnd + 1);
		}
		// Save new value at the end of buffer
		iowDev->buffer[iowDev->bufEnd] = value;
	} else {
		// Insert value at the end of buffer
		iowDev->buffer[iowDev->bufEnd] = value;
		// Set new buffer end position
		iowDev->bufEnd = IOWKIT_BUFFER_POS(iowDev->bufEnd + 1);
		iowDev->numBuf++;
	}
	// Save last read value
	iowDev->lastValue = value;

	return bOk;
}

VOID
IowKitGetBuffer(PIOWKIT_DEVICE_INTERNAL iowDev, DWORD *value)
{
	// Fail if buffer is empty
	if (iowDev->numBuf == 0) {
		return;
	}
	// Copy buffered value
	*value = iowDev->buffer[iowDev->bufStart];
	// Set new buffer start
	iowDev->bufStart = IOWKIT_BUFFER_POS(iowDev->bufStart + 1);
	// Decrement number of buffered values
	iowDev->numBuf--;
	// Reset start & end positions
	if (iowDev->numBuf == 0) {
		iowDev->bufStart = iowDev->bufEnd = 0;
	}
}

// IowKitRead(): read value from device
ULONG IOWKIT_API
IowKitRead(IOWKIT_HANDLE devHandle, ULONG numPipe,
	PCHAR buffer, ULONG length)
{
	PIOWKIT_DEVICE_INTERNAL iowDev;
	HANDLE iowHandle;
	DWORD rc, value;
	ULONG timeout;
	BOOL bOK;
	OVERLAPPED ovlp;

	// Get iowDev
	iowDev = (PIOWKIT_DEVICE_INTERNAL) devHandle;
	// Check for bogus handle
	if (iowDev == NULL) {
		return 0;
	}
	__try {
	// Check cookie
	if (iowDev->cookie != IOWKIT_DEVICE_COOKIE) {
		DbgOut(DBG_ALL, ("Bad cookie?\n"));

		return 0;
	}
	}
	__except ( EXCEPTION_EXECUTE_HANDLER ) {
		DbgOut(DBG_ALL, ("Bad handle, %p\n", iowDev));

		return 0;
	}
	// Get pipe handle
	iowHandle = IowKitGetPipeHandle(iowDev, numPipe);
	// Check if pipe is valid
	if (iowHandle == INVALID_HANDLE_VALUE) {
		SetLastError(ERROR_INVALID_PARAMETER);

		return 0;
	}
	DbgOutF(DBG_ALL, (dbgf, "Reading %d %x %d\n", numPipe, buffer, length));
	ZeroMemory(&ovlp, sizeof(ovlp));
	// Get timeout value
	timeout = iowDev->ioTimeout;
	// Read file asynchronously
	// NOTE: we check if we read from first pipe here and sync with
	// thread in that case
	if ((numPipe == 0) && (iowDev->openMode == IOW_OPEN_SIMPLE)) {
		// Set report ID
		*buffer = 0;
		// Synchronize with thread
		while (1) {
		// For now we're waiting with timeout
		rc = WaitForSingleObject(iowDev->readEvent, timeout);
		// Timeout expired, return
		if (rc != WAIT_OBJECT_0) {
			// Set error code
			SetLastError(ERROR_TIMEOUT);

			return 0;
		}
		// Acquire mutex and get value from buffer
		rc = WaitForSingleObject(iowDev->bufMutex, INFINITE);
		// Check if buffer is empty
		if (iowDev->numBuf == 0) {
			ResetEvent(iowDev->readEvent);
			ReleaseMutex(iowDev->bufMutex);
			continue;
		}
		// Get value from buffer
		IowKitGetBuffer(iowDev, &value);
		break;
		}
		// Copy value
		memcpy(buffer + 1, &value, 4);
		// Reset event until next read happens
		if (iowDev->numBuf == 0) {
			ResetEvent(iowDev->readEvent);
		}
		// Release mutex and exit
		ReleaseMutex(iowDev->bufMutex);

		return length;
	} else {
		// Read from other pipes
		// We don't serialize access to endpoints
		// So, reads always come in some order
		bOK = ReadFileEx(iowHandle, buffer, length, &ovlp, &IowDummyCompletion);
		// Check for error
		// and exit if something is wrong
		if (!bOK && (GetLastError() != ERROR_IO_PENDING)) {
			DbgOut(DBG_ALL, ("IowKitRead(): failed to read\n"));

			return 0;
		}
		// Sleep waiting for I/O completion
		// TODO: make this MsgWaitForMultipleObjectsEx()?
		rc = SleepEx(timeout, TRUE);
		// If rc is WAIT_IO_COMPLETION we read some data
		if (rc == WAIT_IO_COMPLETION) {

			return length;
		}
		// Cancel pending I/O anyway
		CancelIo(iowHandle);
		// Return 0 in case of error or timeout

		return 0;
	}
}

DWORD __stdcall
IowKitReadThread(PIOWKIT_DEVICE_INTERNAL iowHandle)
{
	// Buffer for reading
	union {
		UCHAR buffer[5];
		struct {
			UCHAR reportId;
			DWORD value;
		} r;
	} u;
	// Handle
	HANDLE handle;
	// Set up ovlp
	OVERLAPPED ovlp;
	// SleepEx return value
	DWORD rc;

	// Set up overlapped structure
	ZeroMemory(&ovlp, sizeof(ovlp));
	// Get zero pipe handle
	handle = IowKitGetPipeHandle(iowHandle, 0);
	// Fire up reading
	while (1) {
		// Read asynchronously
		ReadFileEx(handle,
				&u, 5, &ovlp, &IowDummyCompletion);
		// Pefrorm async read
		while (1) {
			// Wait for either I/O completion or for
			// "terminate thread" event
			rc = WaitForSingleObjectEx(iowHandle->endEvent,
							250, TRUE);
			// OK, we read something
			if (rc == WAIT_IO_COMPLETION) {
				// Copy data to buffer
				// lock data
				rc = WaitForSingleObject(iowHandle->bufMutex, INFINITE);
				// We have read something
				iowHandle->readAny = TRUE;
				// Copy new data
				IowKitInsertBuffer(iowHandle, u.r.value);
				// Release mutex
				ReleaseMutex(iowHandle->bufMutex);
				// OK, set event
				SetEvent(iowHandle->readEvent);
				break;
			} else
			// OK, we gotta terminate thread
			// AND tell the program it's safe to close everything
			if (rc == WAIT_OBJECT_0) {
				// Don't forget to cancel all I/O
				CancelIo(handle);
				// Signal that thread is now gone
				SetEvent(iowHandle->threadTermEvent);
				// Goodbye, cruel world!
				ExitThread(0);
			}
		}
	}
}

// Fire up async read thread
BOOLEAN
IowKitFireUpThread(PIOWKIT_DEVICE_INTERNAL iowHandle)
{
	HANDLE thread;
	HANDLE event, mutex;
	DWORD threadId;

	// Create synchronization events
	// Read complete event
	event = CreateEvent(NULL, TRUE, FALSE, NULL);
	// Fail if error
	if (event == NULL) {
		DbgOut(DBG_ALL, ("IowKitFireUpThread(): readEvent failed\n"));
		goto cleanup;
	}
	// Save event handle
	iowHandle->readEvent = event;
	// Read complete event
	event = CreateEvent(NULL, TRUE, FALSE, NULL);
	// Fail if error
	if (event == NULL) {
		DbgOut(DBG_ALL, ("IowKitFireUpThread(): readEvent1 failed\n"));
		goto cleanup;
	}
	// Save event handle
	iowHandle->readEvent1 = event;
	// thread termination event
	event = CreateEvent(NULL, FALSE, FALSE, NULL);
	// Fail if error
	if (event == NULL) {
		DbgOut(DBG_ALL, ("IowKitFireUpThread(): threadterm failed\n"));
		goto cleanup;
	}
	// Save event handle
	iowHandle->endEvent = event;
	// thread has been terminated event
	event = CreateEvent(NULL, FALSE, FALSE, NULL);
	// Fail if error
	if (event == NULL) {
		DbgOut(DBG_ALL, ("IowKitFireUpThread(): threadterm2 failed\n"));
		goto cleanup;
	}
	// Save event handle
	iowHandle->threadTermEvent = event;
	// Create sync mutex
	mutex = CreateMutex(NULL, FALSE, NULL);
	// Fail if error
	if (mutex == NULL) {
		DbgOut(DBG_ALL, ("IowKitFireUpThread(): mutex creation failed\n"));
		goto cleanup;
	}
	// Save mutex handle
	iowHandle->bufMutex = mutex;
	// Create thread in suspended state
	thread = CreateThread(NULL, 0,
				(LPTHREAD_START_ROUTINE) &IowKitReadThread,
				iowHandle,
				CREATE_SUSPENDED,
				&threadId
				);
	// Check if thread is created
	if (thread == NULL) {
			DbgOut(DBG_ALL, ("IowKitFireUpThread(): thread creation failed\n"));
			goto cleanup;
	}
	iowHandle->ioThread = thread;
	// Resume thread
	ResumeThread(thread);

	return TRUE;
cleanup:
	// Cleanup on failure
	if (iowHandle->readEvent) {
		CloseHandle(iowHandle->readEvent);
	}
	if (iowHandle->readEvent1) {
		CloseHandle(iowHandle->readEvent1);
	}
	if (iowHandle->endEvent) {
		CloseHandle(iowHandle->endEvent);
	}
	if (iowHandle->threadTermEvent) {
		CloseHandle(iowHandle->threadTermEvent);
	}
	if (iowHandle->bufMutex) {
		CloseHandle(iowHandle->bufMutex);
	}

	return FALSE;
}

BOOLEAN
IowKitReadNow(PIOWKIT_DEVICE_INTERNAL iowDev, PDWORD value)
{
	HANDLE pipeHandle;
	CHAR report[8];
	BOOLEAN bOK;
	ULONG rc;
	OVERLAPPED ovlp;

	__try {
	// Check handle
	if (iowDev->cookie != IOWKIT_DEVICE_COOKIE) {
		return FALSE;
	}
	// Get 2nd pipe handle
	pipeHandle = IowKitGetPipeHandle(iowDev, 1);
	// Fail if no second pipe
	if (pipeHandle == NULL) {
		return FALSE;
	}
	// Now set up $FF report to read immediate value
	report[0] = (CHAR) 0xFF;
	ZeroMemory(&ovlp, sizeof(ovlp));
	// Check this
	ovlp.hEvent = INVALID_HANDLE_VALUE;
	// Write report
	// TODO: timeout
	bOK = WriteFileEx(pipeHandle, &report, sizeof(report), &ovlp, &IowDummyCompletion);
	// Fail if we couldn't write
	if (!bOK && (GetLastError() != ERROR_IO_PENDING)) {
		// Cancel I/O
		CancelIo(pipeHandle);

		return FALSE;
	}
	// Wait for completion
	rc = SleepEx(1000, TRUE);
	// We failed to write, exit
	if (rc != WAIT_IO_COMPLETION) {
		// Cancel pending I/O
		CancelIo(pipeHandle);
		// Set error
		SetLastError(ERROR_TIMEOUT);

		return FALSE;
	}
	// Now read reply from IOW
	bOK = ReadFileEx(pipeHandle, &report, sizeof(report), &ovlp, &IowDummyCompletion);
	// Wait for completion
	if (!bOK && (GetLastError() != ERROR_IO_PENDING)) {
		// Cancel I/O
		CancelIo(pipeHandle);

		return FALSE;
	}
	// Wait for completion
	rc = SleepEx(1000, TRUE);
	// We failed to write, exit
	if (rc != WAIT_IO_COMPLETION) {
		// Cancel pending I/O
		CancelIo(pipeHandle);
		// Set error
		SetLastError(ERROR_TIMEOUT);

		return FALSE;
	}
	// Copy value back
	*value = *((PDWORD)(&report[1]));

	return TRUE;
	}
	__except (EXCEPTION_EXECUTE_HANDLER) {
		SetLastError(ERROR_INVALID_PARAMETER);

		return FALSE;
	}
}

BOOLEAN
IowKitReadNowLegacy(PIOWKIT_DEVICE_INTERNAL iowDev, PDWORD value)
{
	DWORD val;

	__try {
	// Check and get handle
	if (iowDev == NULL || (iowDev->cookie != IOWKIT_DEVICE_COOKIE)) {
		// Wrong HANDLE!
		return FALSE;
	}
	}
	__except (EXCEPTION_EXECUTE_HANDLER) {
		return FALSE;
	}
	// Lock value
	WaitForSingleObject(iowDev->bufMutex, INFINITE);
	// Fail if we haven't read anything yet
	if (!iowDev->readAny) {
		ReleaseMutex(iowDev->bufMutex);

		return FALSE;
	}
	// Save value
	val = iowDev->lastValue;
	// Unlock value
	ReleaseMutex(iowDev->bufMutex);
	// OK, go ahead and copy value back
	__try {
		*value = val;
	}
	__except (EXCEPTION_EXECUTE_HANDLER) {
		SetLastError(ERROR_INVALID_PARAMETER);

		return FALSE;
	}

	return TRUE;
}

// IowKitReadImmediate(): get last 32-bits value from device
BOOLEAN IOWKIT_API
IowKitReadImmediate(IOWKIT_HANDLE devHandle, PDWORD value)
{
	PIOWKIT_DEVICE_INTERNAL iowDev = (PIOWKIT_DEVICE_INTERNAL) devHandle;
	DWORD val;
	BOOLEAN rc;

	__try {
	// Check and get handle
	if (iowDev == NULL || (iowDev->cookie != IOWKIT_DEVICE_COOKIE)) {
		// Wrong HANDLE!
		return FALSE;
	}
	// Legacy read
	if (iowDev->isLegacy) {
		// Fail if openMode is not supported
		if (iowDev->openMode != IOW_OPEN_SIMPLE) {
			SetLastError(ERROR_INVALID_PARAMETER);

			return FALSE;
		}
		// "Old way" read immedate
		rc = IowKitReadNowLegacy(iowDev, &val);
		// Copy value to user buffer
		*value = val;

		return rc;
	} else {
		// "New way" read immediate
		rc = IowKitReadNow(iowDev, &val);
		// Copy value to user buffer
		*value = val;

		return rc;
	}
	}
	__except (EXCEPTION_EXECUTE_HANDLER) {
		SetLastError(ERROR_INVALID_PARAMETER);

		return FALSE;
	}

	return rc;
}
